import { admin } from "../../firebase";
// Initialize Firestore
const db = admin.firestore();
export const logsCollection = db.collection("activityLogs");

export interface Log {
  id?: string;
  route?: string;
  method?: string;
  requestBody?: Record<string, any>;
  request?: Record<string, any>;
  response?: Record<string, any>;
  responseStatus?: number;
  responseHeaders?: Record<string, any>;
  responseBody?: Record<string, any>;
  requestParams?: Record<string, any>;
  requestQuery?: Record<string, any>;
  requestHeaders?: Record<string, any>;
  requestIp?: string;
  deleted?: boolean;
  createdAt?: FirebaseFirestore.Timestamp;
  updatedAt?: FirebaseFirestore.Timestamp;
}

// ✅ Create a new log
export const createLog = async (data: Log) => {
  const now = admin.firestore.Timestamp.now();
  const logData = {
    ...data,
    deleted: data.deleted ?? false,
    createdAt: now,
    updatedAt: now,
  };
  const docRef = await logsCollection.add(logData);
  return { id: docRef.id, ...logData };
};

// ✅ Get a log by ID
export const getLogById = async (id: string) => {
  const doc = await logsCollection.doc(id).get();
  if (!doc.exists) return null;
  return { id: doc.id, ...doc.data() } as Log;
};

// ✅ Update a log
export const updateLog = async (id: string, updates: Partial<Log>) => {
  await logsCollection.doc(id).update({
    ...updates,
    updatedAt: admin.firestore.Timestamp.now(),
  });
  return getLogById(id);
};

// ✅ Delete a log (soft delete optional)
export const deleteLog = async (id: string, softDelete = true) => {
  if (softDelete) {
    await logsCollection.doc(id).update({
      deleted: true,
      updatedAt: admin.firestore.Timestamp.now(),
    });
  } else {
    await logsCollection.doc(id).delete();
  }
  return true;
};
