import { Request, Response, NextFunction } from "express";
import { BadRequestResponse } from "../utils/ApiResponse";
import { createLog } from "../model/activity.model"; // ✅ updated import

type AsyncFunction = (req: Request, res: Response, next: NextFunction) => Promise<any>;

const asyncHandler = (execution: AsyncFunction) => async (
  req: any,
  res: any,
  next: NextFunction
) => {
  try {
    // ✅ Create log entry before running the route
    const log = await createLog({
      route: req.originalUrl,
      method: req.method,
      requestBody: req.body,
      requestParams: req.params,
      requestQuery: req.query,
      requestHeaders: req.headers,
      requestIp: req.ip,
      responseStatus: res.statusCode,
      responseHeaders: res.getHeaders ? res.getHeaders() : {},
    });

    // Attach log ID to request/response
    req.logId = log.id;
    res.logId = log.id;

    // ✅ Execute the actual controller function
    await execution(req, res, next);
  } catch (error: any) {
    // ✅ Handle any async errors gracefully
    console.error("Async handler error:", error);
    new BadRequestResponse(error?.message || "Unexpected error").send(res);
  }
};

export default asyncHandler;
