import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { SuccessResponse } from "../../utils/ApiResponse";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase";

export default class FluidsController {
  
  saveFluids = asyncHandler(async (req: Request, res: Response) => {
    const { logId } = req.params;
    const { fluidsData, authDetails, createdBy } = req.body;

    if (!fluidsData || !createdBy) {
      throw new BadRequestError("fluidsData and createdBy are required");
    }

    const logRef = db.collection("logs").doc(logId);
    const logDoc = await logRef.get();
    if (!logDoc.exists) {
      throw new BadRequestError("Log not found");
    }

    const logData = logDoc.data()!;
    const logPageNo = logData.logPageNo;

    // Check if logPageNo is undefined and provide a default value
    const safeLogPageNo = logPageNo || ''; // or use null or a default value

    const fluidsDocData = {
      logId,
      logPageNo: safeLogPageNo, // Use the safe value
      fluidsData,
      authDetails: authDetails || {},
      createdBy,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    };

    const existingFluidsSnap = await db
      .collection("fluids")
      .where("logId", "==", logId)
      .get();

    let fluidsId;
    if (existingFluidsSnap.empty) {
      const fluidsRef = db.collection("fluids").doc();
      fluidsId = fluidsRef.id;
      await fluidsRef.set(fluidsDocData);
    } else {
      fluidsId = existingFluidsSnap.docs[0].id;
      await db.collection("fluids").doc(fluidsId).update({
        logPageNo: safeLogPageNo, // Include logPageNo in update as well
        fluidsData,
        authDetails: authDetails || {},
        updatedBy: createdBy,
        updatedAt: admin.firestore.FieldValue.serverTimestamp(),
      });
    }

    // Return direct object response like FlightController
    return res.status(200).json({ 
      success: true,
      message: "Fluids data saved successfully ✅", 
      data: { fluidsId, logId, logPageNo: safeLogPageNo }
    });
  });

  getFluids = asyncHandler(async (req: Request, res: Response) => {
    const { logId } = req.params;

    const fluidsSnap = await db
      .collection("fluids")
      .where("logId", "==", logId)
      .get();

    if (fluidsSnap.empty) {
      // Return direct object response
      return res.status(200).json({ 
        success: true,
        message: "Fluids data fetched",
        data: {
          fluidsData: null,
          authDetails: {}
        }
      });
    }

    const fluidsData = fluidsSnap.docs[0].data();
    
    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "Fluids data fetched successfully",
      data: {
        fluidsId: fluidsSnap.docs[0].id,
        ...fluidsData,
      }
    });
  });
}