/**
 * Cloud functions to handle events from Google Cloud Identity Platform.
 * @packageDocumentation
 */
import { BlockingFunction } from '../../cloud-functions';
import { AuthBlockingEvent, AuthBlockingEventType, AuthUserRecord, BeforeCreateResponse, BeforeSignInResponse, HttpsError } from '../../common/providers/identity';
import * as options from '../options';
import { Expression } from '../params';
export { AuthUserRecord, AuthBlockingEvent, HttpsError };
/** @hidden Internally used when parsing the options. */
interface InternalOptions {
    opts: options.GlobalOptions;
    idToken: boolean;
    accessToken: boolean;
    refreshToken: boolean;
}
/**
 * All function options plus idToken, accessToken, and refreshToken.
 */
export interface BlockingOptions {
    /** Pass the ID Token credential to the function. */
    idToken?: boolean;
    /** Pass the Access Token credential to the function. */
    accessToken?: boolean;
    /** Pass the Refresh Token credential to the function. */
    refreshToken?: boolean;
    /**
     * Region where functions should be deployed.
     */
    region?: options.SupportedRegion | string;
    /**
     * Amount of memory to allocate to a function.
     * A value of null restores the defaults of 256MB.
     */
    memory?: options.MemoryOption | Expression<number> | null;
    /**
     * Timeout for the function in sections, possible values are 0 to 540.
     * HTTPS functions can specify a higher timeout.
     * A value of null restores the default of 60s
     * The minimum timeout for a gen 2 function is 1s. The maximum timeout for a
     * function depends on the type of function: Event handling functions have a
     * maximum timeout of 540s (9 minutes). HTTPS and callable functions have a
     * maximum timeout of 36,00s (1 hour). Task queue functions have a maximum
     * timeout of 1,800s (30 minutes)
     */
    timeoutSeconds?: number | Expression<number> | null;
    /**
     * Min number of actual instances to be running at a given time.
     * Instances will be billed for memory allocation and 10% of CPU allocation
     * while idle.
     * A value of null restores the default min instances.
     */
    minInstances?: number | Expression<number> | null;
    /**
     * Max number of instances to be running in parallel.
     * A value of null restores the default max instances.
     */
    maxInstances?: number | Expression<number> | null;
    /**
     * Number of requests a function can serve at once.
     * Can only be applied to functions running on Cloud Functions v2.
     * A value of null restores the default concurrency (80 when CPU >= 1, 1 otherwise).
     * Concurrency cannot be set to any value other than 1 if `cpu` is less than 1.
     * The maximum value for concurrency is 1,000.
     */
    concurrency?: number | Expression<number> | null;
    /**
     * Fractional number of CPUs to allocate to a function.
     * Defaults to 1 for functions with <= 2GB RAM and increases for larger memory sizes.
     * This is different from the defaults when using the gcloud utility and is different from
     * the fixed amount assigned in Google Cloud Functions generation 1.
     * To revert to the CPU amounts used in gcloud or in Cloud Functions generation 1, set this
     * to the value "gcf_gen1"
     */
    cpu?: number | 'gcf_gen1';
    /**
     * Connect cloud function to specified VPC connector.
     * A value of null removes the VPC connector
     */
    vpcConnector?: string | null;
    /**
     * Egress settings for VPC connector.
     * A value of null turns off VPC connector egress settings
     */
    vpcConnectorEgressSettings?: options.VpcEgressSetting | null;
    /**
     * Specific service account for the function to run as.
     * A value of null restores the default service account.
     */
    serviceAccount?: string | null;
    /**
     * Ingress settings which control where this function can be called from.
     * A value of null turns off ingress settings.
     */
    ingressSettings?: options.IngressSetting | null;
    /**
     * User labels to set on the function.
     */
    labels?: Record<string, string>;
    secrets?: string[];
}
/**
 * Handles an event that is triggered before a user is created.
 * @param handler - Event handler which is run every time before a user is created
 */
export declare function beforeUserCreated(handler: (event: AuthBlockingEvent) => BeforeCreateResponse | Promise<BeforeCreateResponse> | void | Promise<void>): BlockingFunction;
/**
 * Handles an event that is triggered before a user is created.
 * @param opts - Object containing function options
 * @param handler - Event handler which is run every time before a user is created
 */
export declare function beforeUserCreated(opts: BlockingOptions, handler: (event: AuthBlockingEvent) => BeforeCreateResponse | Promise<BeforeCreateResponse> | void | Promise<void>): BlockingFunction;
/**
 * Handles an event that is triggered before a user is signed in.
 * @param handler - Event handler which is run every time before a user is signed in
 */
export declare function beforeUserSignedIn(handler: (event: AuthBlockingEvent) => BeforeSignInResponse | Promise<BeforeSignInResponse> | void | Promise<void>): BlockingFunction;
/**
 * Handles an event that is triggered before a user is signed in.
 * @param opts - Object containing function options
 * @param handler - Event handler which is run every time before a user is signed in
 */
export declare function beforeUserSignedIn(opts: BlockingOptions, handler: (event: AuthBlockingEvent) => BeforeSignInResponse | Promise<BeforeSignInResponse> | void | Promise<void>): BlockingFunction;
/** @hidden */
export declare function beforeOperation(eventType: AuthBlockingEventType, optsOrHandler: BlockingOptions | ((event: AuthBlockingEvent) => BeforeCreateResponse | BeforeSignInResponse | void | Promise<BeforeCreateResponse> | Promise<BeforeSignInResponse> | Promise<void>), handler: (event: AuthBlockingEvent) => BeforeCreateResponse | BeforeSignInResponse | void | Promise<BeforeCreateResponse> | Promise<BeforeSignInResponse> | Promise<void>): BlockingFunction;
/** @hidden */
export declare function getOpts(blockingOptions: BlockingOptions): InternalOptions;
