/**
 * Cloud functions to handle Firebase App Distribution events from Firebase Alerts.
 * @packageDocumentation
 */
import { CloudEvent, CloudFunction } from '../../core';
import * as options from '../../options';
import { Expression } from '../../params';
import { FirebaseAlertData } from './alerts';
/**
 * The internal payload object for adding a new tester device to app distribution.
 * Payload is wrapped inside a `FirebaseAlertData` object.
 */
export interface NewTesterDevicePayload {
    ['@type']: 'type.googleapis.com/google.events.firebase.firebasealerts.v1.AppDistroNewTesterIosDevicePayload';
    /** Name of the tester */
    testerName: string;
    /** Email of the tester */
    testerEmail: string;
    /** The device model name */
    testerDeviceModelName: string;
    /** The device ID */
    testerDeviceIdentifier: string;
}
/**
 * The internal payload object for receiving in-app feedback from a tester.
 * Payload is wrapped inside a `FirebaseAlertData` object.
 *
 * @alpha
 */
export interface InAppFeedbackPayload {
    ['@type']: 'type.googleapis.com/google.events.firebase.firebasealerts.v1.AppDistroInAppFeedbackPayload';
    /** Resource name. Format: `projects/{project_number}/apps/{app_id}/releases/{release_id}/feedbackReports/{feedback_id}` */
    feedbackReport: string;
    /** Deep link back to the Firebase console. */
    feedbackConsoleUri: string;
    /** Name of the tester */
    testerName?: string;
    /** Email of the tester */
    testerEmail: string;
    /**
     * Version consisting of `versionName` and `versionCode` for Android and
     * `CFBundleShortVersionString` and `CFBundleVersion` for iOS.
     */
    appVersion: string;
    /** Text entered by the tester */
    text: string;
    /** URI to download screenshot. This URI is fast expiring. */
    screenshotUri?: string;
}
/**
 * A custom CloudEvent for Firebase Alerts (with custom extension attributes).
 * @typeParam T - the data type for app distribution alerts that is wrapped in a `FirebaseAlertData` object.
 */
export interface AppDistributionEvent<T> extends CloudEvent<FirebaseAlertData<T>> {
    /** The type of the alerts that got triggered. */
    alertType: string;
    /** The Firebase App ID that’s associated with the alert. */
    appId: string;
}
/**
 * Configuration for app distribution functions.
 */
export interface AppDistributionOptions extends options.EventHandlerOptions {
    /** Scope the function to trigger on a specific application. */
    appId?: string;
    /**
     * Region where functions should be deployed.
     */
    region?: options.SupportedRegion | string;
    /**
     * Amount of memory to allocate to a function.
     * A value of null restores the defaults of 256MB.
     */
    memory?: options.MemoryOption | Expression<number> | null;
    /**
     * Timeout for the function in sections, possible values are 0 to 540.
     * HTTPS functions can specify a higher timeout.
     * A value of null restores the default of 60s
     * The minimum timeout for a gen 2 function is 1s. The maximum timeout for a
     * function depends on the type of function: Event handling functions have a
     * maximum timeout of 540s (9 minutes). HTTPS and callable functions have a
     * maximum timeout of 36,00s (1 hour). Task queue functions have a maximum
     * timeout of 1,800s (30 minutes)
     */
    timeoutSeconds?: number | Expression<number> | null;
    /**
     * Min number of actual instances to be running at a given time.
     * Instances will be billed for memory allocation and 10% of CPU allocation
     * while idle.
     * A value of null restores the default min instances.
     */
    minInstances?: number | Expression<number> | null;
    /**
     * Max number of instances to be running in parallel.
     * A value of null restores the default max instances.
     */
    maxInstances?: number | Expression<number> | null;
    /**
     * Number of requests a function can serve at once.
     * Can only be applied to functions running on Cloud Functions v2.
     * A value of null restores the default concurrency (80 when CPU >= 1, 1 otherwise).
     * Concurrency cannot be set to any value other than 1 if `cpu` is less than 1.
     * The maximum value for concurrency is 1,000.
     */
    concurrency?: number | Expression<number> | null;
    /**
     * Fractional number of CPUs to allocate to a function.
     * Defaults to 1 for functions with <= 2GB RAM and increases for larger memory sizes.
     * This is different from the defaults when using the gcloud utility and is different from
     * the fixed amount assigned in Google Cloud Functions generation 1.
     * To revert to the CPU amounts used in gcloud or in Cloud Functions generation 1, set this
     * to the value "gcf_gen1"
     */
    cpu?: number | 'gcf_gen1';
    /**
     * Connect cloud function to specified VPC connector.
     * A value of null removes the VPC connector
     */
    vpcConnector?: string | null;
    /**
     * Egress settings for VPC connector.
     * A value of null turns off VPC connector egress settings
     */
    vpcConnectorEgressSettings?: options.VpcEgressSetting | null;
    /**
     * Specific service account for the function to run as.
     * A value of null restores the default service account.
     */
    serviceAccount?: string | null;
    /**
     * Ingress settings which control where this function can be called from.
     * A value of null turns off ingress settings.
     */
    ingressSettings?: options.IngressSetting | null;
    /**
     * User labels to set on the function.
     */
    labels?: Record<string, string>;
    secrets?: string[];
    /** Whether failed executions should be delivered again. */
    retry?: boolean;
}
/**
 * Declares a function that can handle adding a new tester iOS device.
 * @param handler - Event handler which is run every time a new tester iOS device is added.
 * @returns A function that you can export and deploy.
 */
export declare function onNewTesterIosDevicePublished(handler: (event: AppDistributionEvent<NewTesterDevicePayload>) => any | Promise<any>): CloudFunction<AppDistributionEvent<NewTesterDevicePayload>>;
/**
 * Declares a function that can handle adding a new tester iOS device.
 * @param appId - A specific application the handler will trigger on.
 * @param handler - Event handler which is run every time a new tester iOS device is added.
 * @returns A function that you can export and deploy.
 */
export declare function onNewTesterIosDevicePublished(appId: string, handler: (event: AppDistributionEvent<NewTesterDevicePayload>) => any | Promise<any>): CloudFunction<AppDistributionEvent<NewTesterDevicePayload>>;
/**
 * Declares a function that can handle adding a new tester iOS device.
 * @param opts - Options that can be set on the function.
 * @param handler - Event handler which is run every time a new tester iOS device is added.
 * @returns A function that you can export and deploy.
 */
export declare function onNewTesterIosDevicePublished(opts: AppDistributionOptions, handler: (event: AppDistributionEvent<NewTesterDevicePayload>) => any | Promise<any>): CloudFunction<AppDistributionEvent<NewTesterDevicePayload>>;
/**
 * Declares a function that can handle receiving new in-app feedback from a tester.
 * @param handler - Event handler which is run every time new feedback is received.
 * @returns A function that you can export and deploy.
 *
 * @alpha
 */
export declare function onInAppFeedbackPublished(handler: (event: AppDistributionEvent<InAppFeedbackPayload>) => any | Promise<any>): CloudFunction<AppDistributionEvent<InAppFeedbackPayload>>;
/**
 * Declares a function that can handle receiving new in-app feedback from a tester.
 * @param appId - A specific application the handler will trigger on.
 * @param handler - Event handler which is run every time new feedback is received.
 * @returns A function that you can export and deploy.
 *
 * @alpha
 */
export declare function onInAppFeedbackPublished(appId: string, handler: (event: AppDistributionEvent<InAppFeedbackPayload>) => any | Promise<any>): CloudFunction<AppDistributionEvent<InAppFeedbackPayload>>;
/**
 * Declares a function that can handle receiving new in-app feedback from a tester.
 * @param opts - Options that can be set on the function.
 * @param handler - Event handler which is run every time new feedback is received.
 * @returns A function that you can export and deploy.
 *
 * @alpha
 */
export declare function onInAppFeedbackPublished(opts: AppDistributionOptions, handler: (event: AppDistributionEvent<InAppFeedbackPayload>) => any | Promise<any>): CloudFunction<AppDistributionEvent<InAppFeedbackPayload>>;
