import { Expression } from './params';
import { ParamSpec } from './params/types';
/**
 * List of all regions supported by Cloud Functions v2
 */
export declare type SupportedRegion = 'asia-northeast1' | 'europe-north1' | 'europe-west1' | 'europe-west4' | 'us-central1' | 'us-east1' | 'us-west1';
/**
 * List of available memory options supported by Cloud Functions.
 */
export declare type MemoryOption = '128MiB' | '256MiB' | '512MiB' | '1GiB' | '2GiB' | '4GiB' | '8GiB' | '16GiB' | '32GiB';
/**
 * List of available options for VpcConnectorEgressSettings.
 */
export declare type VpcEgressSetting = 'PRIVATE_RANGES_ONLY' | 'ALL_TRAFFIC';
/**
 * List of available options for IngressSettings.
 */
export declare type IngressSetting = 'ALLOW_ALL' | 'ALLOW_INTERNAL_ONLY' | 'ALLOW_INTERNAL_AND_GCLB';
/**
 * GlobalOptions are options that can be set across an entire project.
 * These options are common to HTTPS and Event handling functions.
 */
export interface GlobalOptions {
    /**
     * Region where functions should be deployed.
     * HTTP functions can override and specify more than one region.
     */
    region?: SupportedRegion | string;
    /**
     * Amount of memory to allocate to a function.
     * A value of null restores the defaults of 256MB.
     */
    memory?: MemoryOption | Expression<number> | null;
    /**
     * Timeout for the function in sections, possible values are 0 to 540.
     * HTTPS functions can specify a higher timeout.
     * A value of null restores the default of 60s
     * The minimum timeout for a gen 2 function is 1s. The maximum timeout for a
     * function depends on the type of function: Event handling functions have a
     * maximum timeout of 540s (9 minutes). HTTPS and callable functions have a
     * maximum timeout of 36,00s (1 hour). Task queue functions have a maximum
     * timeout of 1,800s (30 minutes)
     */
    timeoutSeconds?: number | Expression<number> | null;
    /**
     * Min number of actual instances to be running at a given time.
     * Instances will be billed for memory allocation and 10% of CPU allocation
     * while idle.
     * A value of null restores the default min instances.
     */
    minInstances?: number | Expression<number> | null;
    /**
     * Max number of instances to be running in parallel.
     * A value of null restores the default max instances.
     */
    maxInstances?: number | Expression<number> | null;
    /**
     * Number of requests a function can serve at once.
     * Can only be applied to functions running on Cloud Functions v2.
     * A value of null restores the default concurrency (80 when CPU >= 1, 1 otherwise).
     * Concurrency cannot be set to any value other than 1 if `cpu` is less than 1.
     * The maximum value for concurrency is 1,000.
     */
    concurrency?: number | Expression<number> | null;
    /**
     * Fractional number of CPUs to allocate to a function.
     * Defaults to 1 for functions with <= 2GB RAM and increases for larger memory sizes.
     * This is different from the defaults when using the gcloud utility and is different from
     * the fixed amount assigned in Google Cloud Functions generation 1.
     * To revert to the CPU amounts used in gcloud or in Cloud Functions generation 1, set this
     * to the value "gcf_gen1"
     */
    cpu?: number | 'gcf_gen1';
    /**
     * Connect cloud function to specified VPC connector.
     * A value of null removes the VPC connector
     */
    vpcConnector?: string | null;
    /**
     * Egress settings for VPC connector.
     * A value of null turns off VPC connector egress settings
     */
    vpcConnectorEgressSettings?: VpcEgressSetting | null;
    /**
     * Specific service account for the function to run as.
     * A value of null restores the default service account.
     */
    serviceAccount?: string | null;
    /**
     * Ingress settings which control where this function can be called from.
     * A value of null turns off ingress settings.
     */
    ingressSettings?: IngressSetting | null;
    /**
     * User labels to set on the function.
     */
    labels?: Record<string, string>;
    /**
     * Invoker to set access control on https functions.
     */
    invoker?: 'public' | 'private' | string | string[];
    secrets?: string[];
}
/**
 * Sets default options for all functions written using the v2 SDK.
 * @param options Options to set as default
 */
export declare function setGlobalOptions(options: GlobalOptions): void;
/**
 * Additional fields that can be set on any event-handling Cloud Function.
 */
export interface EventHandlerOptions extends GlobalOptions {
    eventType?: string;
    eventFilters?: Record<string, string | Expression<string>>;
    eventFilterPathPatterns?: Record<string, string | Expression<string>>;
    /** Whether failed executions should be delivered again. */
    retry?: boolean | Expression<boolean> | null;
    /** Region of the EventArc trigger. */
    region?: string;
    /** The service account that EventArc should use to invoke this function. Requires the P4SA to have ActAs permission on this service account. */
    serviceAccount?: string | null;
    /** The name of the channel where the function receives events. */
    channel?: string;
}
/**
 * @hidden
 * @alpha
 */
export declare function __getSpec(): {
    globalOptions: GlobalOptions;
    params: ParamSpec[];
};
